﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;

    using Domain.Helpers;
    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Shared.Library.Enums;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;

    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.UserModels.Filters;
    using Utilities;

    /// <summary>
    /// The provider bank account controller.
    /// </summary>
    [Route("api/provider-bank-account")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class ProviderBankAccountController : BaseController
    {
        /// <summary>
        /// the provider service
        /// </summary>
        private readonly IProviderBankAccountService providerBankAccountService;

        /// <summary>
        /// The AES helper.
        /// </summary>
        private readonly IAESHelper aesHelper;

        /// <summary>
        /// The auditlog services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <summary>
        /// the provider service
        /// </summary>
        private readonly IProviderService providerServices;



        /// <inheritdoc />
        public ProviderBankAccountController(IAESHelper aesHelper, IProviderBankAccountService providerBankAccountService, IProviderService providerServices, IAuditLogService auditLogServices)
        {
            this.aesHelper = aesHelper;
            this.providerBankAccountService = providerBankAccountService;
            this.providerServices = providerServices;
            this.auditLogServices = auditLogServices;
        }

        /// <summary>
        /// To find provider
        /// </summary>
        /// <param name="model" >
        /// The provider find request Model.
        /// </param>
        /// <returns>
        /// The provider model.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Provider model.
        /// - 400 - Sorry! We don't have a provider in the system.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("find")]
        [ProducesResponseType(typeof(ProviderModel), 200)]
        [ProducesResponseType(typeof(string), 400)]
        [ProducesResponseType(500)]

        public async Task<ActionResult> FindAsync([FromBody]ProviderBankAccountModel model)
        {
            var provider = await this.providerBankAccountService.FindAsync(model.ProviderId);
            if (provider == null || provider.ProviderId == 0)
            {
                return this.BadRequest("Sorry! We don't have a provider in the system.");
            }

            return this.Success(provider);
        }

        /// <summary>
        /// To fetch providers
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The list of providers.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of providers.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("fetch")]
        [ProducesResponseType(typeof(IEnumerable<ProviderModel>), 200)]
        [ProducesResponseType(500)]

        public async Task<ActionResult> FetchAsync([FromBody] ProviderFilterModel model)
        {
            model = (ProviderFilterModel)EmptyFilter.Handler(model);
            var providers = await this.providerBankAccountService.FetchAsync(model);
            if (providers == null)
            {
                return this.ServerError();
            }

            foreach (var provider in providers)
            {
                provider.EncryptedProviderId = this.aesHelper.Encode(provider.ProviderId.ToString());
            }

            return this.Success(providers);
        }

        /// <summary>
        /// The add provider.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Authorize]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(typeof(string), 409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] ProviderBankAccountModel model, [FromHeader] LocationHeader header)
        {
            var accountId = await this.providerBankAccountService.AddAsync(model);

            var provider = await this.providerServices.FindProviderByProviderId(model.ProviderId);

            switch (accountId)
            {
                case -1:
                    return this.Conflict("Given email address has already been exists with us.");
                case -2:
                    return this.Conflict("Given mobile number has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

        
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.BankDetails,
                LogFrom = (short)model.RoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $"{model.FullName} has added Bank details of a Doctor {provider.FullName} on {DateTime.UtcNow.AddMinutes(330)}",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Bank details added successfully.");
        }

        /// <summary>
        /// The update provider.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Provider details has been updated successfully.
        /// - 409 - Provider has already been exists with us.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Authorize]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(typeof(string), 409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] ProviderBankAccountModel model,[FromHeader] LocationHeader header)
        {
            var provider = await this.providerServices.FindProviderByProviderId(model.ProviderId);
            var response = await this.providerBankAccountService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given email address has already been exists with us.");
                case -2:
                    return this.Conflict("Given mobile number has already been exists with us.");
                case -3:
                    return this.Conflict("Given NPI has already been exists with us.");
                case 0:
                    return this.ServerError();
                default:

                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.ModifiedBy,
                        LogTypeId = (int)LogTypes.BankDetails,
                        LogFrom = (short)model.RoleId,
                        LogDate = DateTime.UtcNow.AddMinutes(330),
                        LogDescription = $"{model.FullName} has Updated Bank details of a Doctor {provider.FullName} on {DateTime.UtcNow.AddMinutes(330)}",
                        LocationId = Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLogModel);

                    return this.Success("Bank details has been updated successfully.");
            }
        }
    }
}